/* Tower of Hanoi — Kindle-tuned, ES5-only (Bigger UI + Win-fix + 8 discs)
   - Fills most of the screen (w≈96%, h≈80–88% of available)
   - Exact pixel layout first, then draw (no overflow)
   - Discs centered via left:50% + negative margin-left
   - Classic 1-disc rule; win only when all discs are on PEG C (target = 2)
   - Refresh overlay (no page reload)
   by kindlemodshelfguy
*/

var PEGS = 3;
var TARGET_PEG = 2;         // require final stack on C (prevents false win on A)
var pegs = [];              // arrays bottom->top; index 0 = bottom
var holding = null;         // { size: n, from: pegIndex } or null
var moves = 0;
var DISC_COUNT = 5;
var MIN_DISCS = 3, MAX_DISCS = 8;

/* Layout cache */
var L = {
  boardW: 480,   // px
  cellW: 160,    // px
  pegW: 140,     // px
  pegH: 300,     // px
  gap: 3,        // px
  discH: 20      // px
};

/* DOM utils */
function $(id){ try{ return document.getElementById(id); }catch(e){ return null; } }
function setText(id, t){ var el=$(id); if(!el) return; try{ el.textContent = t; }catch(e){ el.innerHTML = t; } }
function addClass(el, n){ if(!el) return; if((' '+el.className+' ').indexOf(' '+n+' ')===-1) el.className=(el.className?el.className+' ':'')+n; }
function removeClass(el, n){ if(!el) return; el.className=(' '+el.className+' ').replace(' '+n+' ',' ').replace(/^\s+|\s+$/g,''); }

/* Build board DOM */
function buildBoard(){
  var row = $("pegRow"); row.innerHTML = "";
  var p;
  for(p=0;p<PEGS;p++){
    var td = document.createElement("td");
    td.className = "pegCell";
    td.onclick = (function(idx){ return function(){ pegClick(idx); }; })(p);

    var peg = document.createElement("div");
    peg.className = "peg";
    peg.id = "peg_"+p;

    var rod = document.createElement("div");
    rod.className = "rod";
    rod.id = "rod_"+p;

    var base = document.createElement("div");
    base.className = "pegBase";
    base.id = "pegBase_"+p;

    peg.appendChild(rod);
    td.appendChild(peg);
    td.appendChild(base);
    row.appendChild(td);
  }
}

/* Compute exact pixel layout (bigger but safe) */
function layoutPass(){
  var w = window.innerWidth || document.documentElement.clientWidth || 600;
  var h = window.innerHeight || document.documentElement.clientHeight || 800;

  var top = $("top"), msg = $("message"), ctr = $("controlRow");
  var topH = top ? top.offsetHeight : 0;
  var msgH = msg ? msg.offsetHeight : 0;
  var ctrH = ctr ? ctr.offsetHeight : 0;
  var padding = 16;

  var availH = h - (topH + ctrH + msgH + padding);
  if(availH < 220) availH = 220;

  // Make the board very wide but keep small margins; also bound by height
  var targetW = Math.floor(w * 0.96);
  var maxWByH = Math.floor(h * 0.98);
  if(targetW > maxWByH) targetW = maxWByH;
  if(targetW < 360) targetW = 360;

  L.boardW = targetW;
  L.cellW  = Math.floor(L.boardW / 3);

  // Taller pegs (~82% of available height), but never exceed available space
  L.pegH = Math.floor(availH * 0.82);
  if(L.pegH < 220) L.pegH = 220;
  if(L.pegH > availH - 6) L.pegH = availH - 6;

  // Wider pegs (~90% of their cell) so discs look big
  L.pegW = Math.floor(L.cellW * 0.90);
  if(L.pegW > L.cellW - 8) L.pegW = L.cellW - 8;

  // Disc vertical metrics: fit all discs + gaps cleanly
  L.gap = 3;
  var usableH = L.pegH - 22;
  var gaps = (DISC_COUNT > 1) ? (DISC_COUNT - 1) * L.gap : 0;
  L.discH = Math.floor((usableH - gaps) / DISC_COUNT);
  if(L.discH < 16) L.discH = 16;   // still readable at 8 discs
  if(L.discH > 34) L.discH = 34;

  // Apply pixel sizes
  var wrap = $("boardWrap");
  wrap.style.width = L.boardW + "px";

  var p;
  for(p=0;p<PEGS;p++){
    var pegEl = $("peg_"+p);
    if(pegEl){
      pegEl.style.width  = L.pegW + "px";
      pegEl.style.height = L.pegH + "px";
    }
  }

  // Labels & headers a touch larger
  var labelSize = Math.max(18, Math.floor(L.boardW * 0.060));
  var labs = document.getElementsByClassName("pegLabel");
  var i;
  for(i=0;i<labs.length;i++){ labs[i].style.fontSize = labelSize + "px"; }

  $("title").style.fontSize    = Math.max(22, Math.floor(L.boardW * 0.070)) + "px";
  $("subtitle").style.fontSize = Math.max(12, Math.floor(L.boardW * 0.032)) + "px";
}

/* Start new game */
function newGame(size){
  var s=parseInt(size,10); if(isNaN(s)) s=DISC_COUNT;
  if(s<MIN_DISCS) s=MIN_DISCS; if(s>MAX_DISCS) s=MAX_DISCS;
  DISC_COUNT=s;

  pegs=[]; var p; for(p=0;p<PEGS;p++) pegs[p]=[];
  var d; for(d=DISC_COUNT; d>=1; d--){ pegs[0].push(d); } // bottom->top
  holding=null; moves=0;
  setText("moves", String(moves));
  setText("discCount", String(DISC_COUNT));
  setText("holding","—");
  setText("message","");

  layoutPass();
  draw();
}

/* Draw from computed layout */
function draw(){
  var p;
  for(p=0;p<PEGS;p++){
    var pegEl = $("peg_"+p);
    var rod   = $("rod_"+p);

    while(pegEl.firstChild) pegEl.removeChild(pegEl.firstChild);
    if(rod) pegEl.appendChild(rod);

    var stack = pegs[p]; // bottom->top
    var i;

    // Disc width range (strictly inside peg)
    var minW = Math.floor(L.pegW * 0.32);
    var maxW = Math.floor(L.pegW * 0.96);
    if(minW < 48) minW = 48;
    if(maxW > L.pegW - 6) maxW = L.pegW - 6;

    for(i=0;i<stack.length;i++){
      var size = stack[i]; // 1..DISC_COUNT (1 = smallest)
      var disc = document.createElement("div");
      disc.className = "disc";

      var w = minW + Math.round((size - 1) * (maxW - minW) / Math.max(1, DISC_COUNT - 1));
      if(w > maxW) w = maxW;
      if(w < minW) w = minW;

      disc.style.width = w + "px";
      disc.style.height = L.discH + "px";
      disc.style.marginLeft = (-Math.floor(w/2)) + "px";

      var bottomPos = i * (L.discH + L.gap) + 5;
      disc.style.bottom = bottomPos + "px";

      var shade = 205 - Math.round((size - 1) * 60 / Math.max(1, DISC_COUNT - 1)); // 205..145
      disc.style.background = "rgb(" + shade + "," + shade + "," + shade + ")";
      disc.style.border = "1px solid #777";

      (function(pegIndex){
        disc.onclick = function(e){ try{ if(e && e.stopPropagation) e.stopPropagation(); }catch(x){} pegClick(pegIndex); };
      })(p);

      pegEl.appendChild(disc);
    }

    var td = pegEl.parentNode;
    if(holding && canDropOn(p, holding.size)){ addClass(td,"peg-holding"); }
    else { removeClass(td,"peg-holding"); }
  }

  if(holding){ setText("holding", String(holding.size)); } else { setText("holding","—"); }
}

/* Moves */
function pegClick(pegIndex){
  if(holding){
    if(pegIndex===holding.from){
      pegs[holding.from].push(holding.size);
      holding=null; setText("message","Cancelled"); draw(); return;
    }
    if(canDropOn(pegIndex,holding.size)){
      pegs[pegIndex].push(holding.size);
      holding=null; moves+=1; setText("moves", String(moves)); setText("message","");
      draw();
      if(checkWin()) setText("message","Solved in "+moves+" moves!");
      return;
    } else {
      setText("message","Illegal move"); setTimeout(function(){ setText("message",""); },700);
      return;
    }
  } else {
    var stack=pegs[pegIndex]; if(stack.length===0){ setText("message","Peg empty"); setTimeout(function(){ setText("message",""); },600); return; }
    var top=stack[stack.length-1]; stack.pop();
    holding={size:top, from:pegIndex}; setText("message","Picked disc "+top);
    draw();
  }
}

function canDropOn(pegIndex, moving){
  var st=pegs[pegIndex]; if(st.length===0) return true;
  var top=st[st.length-1]; return top>moving;
}

/* Win only when all discs are stacked (largest->smallest) on TARGET_PEG (C) */
function checkWin(){
  var st = pegs[TARGET_PEG];
  if(!st || st.length !== DISC_COUNT) return false;
  var i;
  for(i=0;i<DISC_COUNT-1;i++){
    if(st[i] <= st[i+1]) return false; // must be strictly decreasing bottom->top
  }
  return true;
}

/* E-ink refresh overlay (no page reload) */
function refreshScreen(){
  var ov=document.createElement("div");
  ov.style.position="fixed"; ov.style.left="0"; ov.style.top="0";
  ov.style.width="100%"; ov.style.height="100%";
  ov.style.background="#ffffff"; ov.style.zIndex="9999";
  document.body.appendChild(ov);
  setTimeout(function(){ if(ov&&ov.parentNode) ov.parentNode.removeChild(ov); }, 140);
}

/* Buttons */
function press(btn, action){
  try{ if(btn&&btn.blur) btn.blur(); }catch(e){}
  if(action==='new'){ var sel=$("sizeSelect"); newGame(sel?sel.value:DISC_COUNT); return; }
  if(action==='undo'){ undo(); return; }
  if(action==='refresh'){ refreshScreen(); return; }
}

function undo(){
  if(holding){
    pegs[holding.from].push(holding.size);
    holding=null; setText("message","Undo pick"); draw(); return;
  }
  setText("message","(Nothing to undo)"); setTimeout(function(){ setText("message",""); },700);
}

function changeSize(val){
  var v=parseInt(val,10); if(isNaN(v)) return;
  if(v<MIN_DISCS) v=MIN_DISCS; if(v>MAX_DISCS) v=MAX_DISCS;
  DISC_COUNT=v; newGame(DISC_COUNT);
}

/* Resize handler */
function resizeLayout(){
  layoutPass();
  draw();
}

/* Init */
function init(){
  buildBoard();
  var sel=$("sizeSelect"); if(sel) sel.value=String(DISC_COUNT);
  layoutPass();
  newGame(DISC_COUNT);
  try{ window.addEventListener("resize", resizeLayout); }catch(e){}
}

/* Expose */
try{ window.press=press; window.changeSize=changeSize; }catch(e){}

/* DOM ready */
if(document.readyState==="loading"){
  try{ document.addEventListener("DOMContentLoaded", init); }catch(e){ window.onload=init; }
}else{ init(); }