#!/bin/sh
# Name: TowerOfHanoi
# Author: kindlemodshelfguy
# Purpose: Always replace the installed WAF with what's in /mnt/us/documents,
#          stopping the running instance first and restarting after install.
# DontUseFBInk

# --- CONFIG ---
SOURCE_DIR="/mnt/us/documents/tower-of-hanoi"
TARGET_DIR="/var/local/mesquite/tower-of-hanoi"
DB="/var/local/appreg.db"
APP_ID="com.kindlemodshelf.towerofhanoi"
# --- end config ---

set -e

# require root
if [ "$(id -u)" != "0" ]; then
  echo "Must run as root"
  exit 1
fi

if [ ! -d "$SOURCE_DIR" ]; then
  echo "Source dir missing: $SOURCE_DIR"
  exit 1
fi

echo "Stopping running app (if any)..."
# ask appmgrd to stop the app (this is the canonical way used earlier)
# ignore errors so script continues even if app wasn't running
lipc-set-prop com.lab126.appmgrd stop app://$APP_ID >/dev/null 2>&1 || true
sleep 1

# attempt to kill leftover mesquite processes that reference this app (best-effort)
if command -v pgrep >/dev/null 2>&1; then
  PIDS=$(pgrep -f "mesquite.*-l $APP_ID" || true)
  if [ -n "$PIDS" ]; then
    echo "Killing leftover mesquite processes: $PIDS"
    kill -9 $PIDS || true
    sleep 1
  fi
fi

# backup existing target (optional — keep 1 backup)
if [ -d "$TARGET_DIR" ]; then
  BACKUP="${TARGET_DIR}_backup_$(date +%Y%m%d_%H%M%S)"
  echo "Backing up existing install to: $BACKUP"
  rm -rf "$BACKUP" || true
  cp -a "$TARGET_DIR" "$BACKUP" || true
fi

echo "Replacing target directory..."
rm -rf "$TARGET_DIR"
cp -a "$SOURCE_DIR" "$TARGET_DIR"
sync

echo "Updating app registration in $DB..."
# Replace properties in appreg.db so we always end up with the correct registration.
# We delete any old properties/handler entries for the handlerId and re-insert.
sqlite3 "$DB" <<EOF
PRAGMA journal_mode=DELETE;
BEGIN TRANSACTION;
DELETE FROM properties WHERE handlerId='$APP_ID';
DELETE FROM handlerIds WHERE handlerId='$APP_ID';
INSERT OR IGNORE INTO interfaces(interface) VALUES('application');
INSERT OR IGNORE INTO handlerIds(handlerId) VALUES('$APP_ID');
INSERT OR REPLACE INTO properties(handlerId,name,value) VALUES('$APP_ID','lipcId','$APP_ID');
INSERT OR REPLACE INTO properties(handlerId,name,value) VALUES('$APP_ID','command','/usr/bin/mesquite -l $APP_ID -c file://$TARGET_DIR/');
INSERT OR REPLACE INTO properties(handlerId,name,value) VALUES('$APP_ID','supportedOrientation','U');
COMMIT;
EOF

sync

echo "Starting app via appmgrd..."
# start the app via app manager (best practice previously used)
nohup lipc-set-prop com.lab126.appmgrd start app://$APP_ID >/dev/null 2>&1 &

echo "Install complete. Installed to: $TARGET_DIR"
exit 0