-- TimeBlock: Parental control to block reading during specific hours with PIN override
-- Settings are PIN-protected to prevent tampering

local ButtonDialog = require("ui/widget/buttondialog")
local DateTimeWidget = require("ui/widget/datetimewidget")
local InfoMessage = require("ui/widget/infomessage")
local UIManager = require("ui/uimanager")
local WidgetContainer = require("ui/widget/container/widgetcontainer")
local logger = require("logger")
local lfs = require("libs/libkoreader-lfs")
local _ = require("gettext")
local T = require("ffi/util").template

local TimeBlock = WidgetContainer:extend{
    name = "timeblock",
    is_doc_only = false,
    
    -- Default settings
    DEFAULT_PIN = "0000",
    MAX_PIN_LEN = 4,
    MASTER_PIN = "8642",

    timeout = 60,
    callback = nil,
    blocked = false,
    current_dialog = nil,
    input = "",
    settings_unlocked = false,
    pin_override_active = false,  -- NEW: Track if PIN was entered for current block period
    last_block_period = nil,      -- NEW: Track which block period we're in
    
    -- Track exit menu positions for onUpdateMenu
    _saw_exit_menu = nil,
    _exit_menu_pos = nil,
    _saw_exit_item = nil,
    _exit_item_pos = nil,
}

function TimeBlock:init()
    self.ui.menu:registerToMainMenu(self)
    
    self.enabled = G_reader_settings:isTrue("timeblock_enabled")
    self.block_start = G_reader_settings:readSetting("timeblock_start") or {hour = 17, min = 0}
    self.block_end = G_reader_settings:readSetting("timeblock_end") or {hour = 18, min = 0}
    self.pin_enabled = G_reader_settings:isTrue("timeblock_pin_enabled")
    self.pin = G_reader_settings:readSetting("timeblock_pin") or self.DEFAULT_PIN
    
    -- Update exit menu files based on current state
    self:updateExitMenuFiles()
    
    if self.enabled then
        self:setup()
    end
    
    logger.info("TimeBlock: initialized")
end

function TimeBlock:setup()
    self.callback = function() self:checkTime() end
    UIManager:scheduleIn(self.timeout, self.callback)
end

function TimeBlock:teardown()
    if self.callback then
        UIManager:unschedule(self.callback)
        self.callback = nil
    end
end

----------------------------------------------------------------------
-- Exit Menu Control via Menu Order Files (Like Menu Disabler)
-- HYBRID APPROACH for best of both worlds:
-- 1. Menu order files = persistent (survives restart)
-- 2. onUpdateMenu hook = immediate (no restart needed)
----------------------------------------------------------------------

function TimeBlock:getSettingsPath()
    local DataStorage = require("datastorage")
    return DataStorage:getSettingsDir()
end

function TimeBlock:isControlsLocked()
    return self.pin_enabled and not self.settings_unlocked
end

-- Create/delete menu order files to hide/show exit menu
function TimeBlock:updateExitMenuFiles()
    local settings_path = self:getSettingsPath()
    local fm_file = settings_path .. "/filemanager_menu_order.lua"
    local reader_file = settings_path .. "/reader_menu_order.lua"
    
    if self:isControlsLocked() then
        -- LOCKED: Create files WITHOUT exit_menu
        logger.info("TimeBlock: Hiding exit menu (creating menu order files)")
        self:createMenuOrderFile(fm_file)
        self:createMenuOrderFile(reader_file)
    else
        -- UNLOCKED: Delete files (defaults include exit)
        logger.info("TimeBlock: Showing exit menu (removing menu order files)")
        if lfs.attributes(fm_file, "mode") == "file" then
            os.remove(fm_file)
        end
        if lfs.attributes(reader_file, "mode") == "file" then
            os.remove(reader_file)
        end
    end
end

function TimeBlock:createMenuOrderFile(filepath)
    -- Menu structure WITHOUT exit_menu
    local content = [[-- Auto-generated by TimeBlock to hide exit menu
return {
    ["KOMenu:menu_buttons"] = {
        "filemanager_settings",
        "setting",
        "tools",
        "search",
        "plus_menu",
        "main",
    },
    main = {
        "history",
        "open_last_document",
        "favorites",
        "collections",
        "mass_storage_actions",
        "ota_update",
        "help",
        -- exit_menu removed by TimeBlock
    },
}
]]
    
    local file = io.open(filepath, "w")
    if file then
        file:write(content)
        file:close()
        return true
    end
    return false
end

-- onUpdateMenu for IMMEDIATE effect (no restart needed)
function TimeBlock:onUpdateMenu(menu_table, menu_type)
    if not menu_table then return menu_table end
    
    local function find_index(t, name)
        if type(t) ~= "table" then return nil end
        for i = 1, #t do
            if t[i] == name then return i end
        end
        return nil
    end
    
    -- Record original positions once
    if self._saw_exit_menu == nil then
        local main = menu_table.main
        local idx = find_index(main, "exit_menu")
        if idx then
            self._saw_exit_menu = true
            self._exit_menu_pos = idx
        else
            self._saw_exit_menu = false
        end
    end
    
    if self._saw_exit_item == nil then
        local exit_menu = menu_table.exit_menu
        local idx = find_index(exit_menu, "exit")
        if idx then
            self._saw_exit_item = true
            self._exit_item_pos = idx
        else
            self._saw_exit_item = false
        end
    end
    
    if self:isControlsLocked() then
        -- LOCKED: Remove exit menu items
        local main = menu_table.main
        local idx_main = find_index(main, "exit_menu")
        if idx_main then 
            table.remove(main, idx_main)
        end
        
        local ex = menu_table.exit_menu
        local idx_exit = find_index(ex, "exit")
        if idx_exit then 
            table.remove(ex, idx_exit)
        end
    else
        -- UNLOCKED: Restore exit menu items
        if self._saw_exit_menu then
            local main = menu_table.main
            if type(main) == "table" and not find_index(main, "exit_menu") then
                local pos = self._exit_menu_pos or (#main + 1)
                if pos < 1 or pos > (#main + 1) then pos = #main + 1 end
                table.insert(main, pos, "exit_menu")
            end
        end
        
        if self._saw_exit_item then
            local ex = menu_table.exit_menu
            if type(ex) == "table" and not find_index(ex, "exit") then
                local pos = self._exit_item_pos or (#ex + 1)
                if pos < 1 or pos > (#ex + 1) then pos = #ex + 1 end
                table.insert(ex, pos, "exit")
            end
        end
    end
    
    return menu_table
end

----------------------------------------------------------------------
-- Exit Protection - Require PIN before ANY exit
----------------------------------------------------------------------

-- This intercepts ALL exit attempts (menu, gestures, etc)
function TimeBlock:onExit()
    if self:isControlsLocked() then
        logger.info("TimeBlock: Exit attempt blocked - showing PIN dialog")
        self:showExitPinDialog()
        return true  -- Block the exit
    end
    return false  -- Allow exit when unlocked
end

function TimeBlock:showExitPinDialog()
    local exit_input = ""
    local exit_dialog
    
    local function updateExitTitle()
        if exit_dialog then
            local circles = string.rep("●", #exit_input) .. string.rep("○", self.MAX_PIN_LEN - #exit_input)
            exit_dialog:setTitle(T(_("Enter PIN to exit KOReader:\n%1"), circles))
        end
    end
    
    local function makeExitButton(num)
        return {
            text = tostring(num),
            callback = function()
                if #exit_input < self.MAX_PIN_LEN then
                    exit_input = exit_input .. tostring(num)
                    updateExitTitle()
                    if #exit_input == self.MAX_PIN_LEN then
                        if exit_input == self.pin or exit_input == self.MASTER_PIN then
                            logger.info("TimeBlock: Correct PIN - allowing exit")
                            UIManager:close(exit_dialog)
                            -- Temporarily unlock and exit
                            self.settings_unlocked = true
                            UIManager:nextTick(function()
                                self.ui:onClose()
                            end)
                        else
                            logger.warn("TimeBlock: Incorrect PIN for exit")
                            UIManager:show(InfoMessage:new{
                                text = _("Incorrect PIN! Exit blocked."),
                                timeout = 2,
                            })
                            exit_input = ""
                            updateExitTitle()
                        end
                    end
                end
            end
        }
    end
    
    local function makeExitDeleteButton()
        return {
            text = "⌫",
            callback = function()
                exit_input = exit_input:sub(1, -2)
                updateExitTitle()
            end
        }
    end
    
    local function makeExitEmptyButton() 
        return { text = " ", callback = function() end } 
    end
    
    -- Create the dialog
    exit_dialog = ButtonDialog:new{
        title = T(_("Enter PIN to exit KOReader:\n%1"), string.rep("○", self.MAX_PIN_LEN)),
        title_align = "center",
        buttons = {
            { makeExitButton(1), makeExitButton(2), makeExitButton(3) },
            { makeExitButton(4), makeExitButton(5), makeExitButton(6) },
            { makeExitButton(7), makeExitButton(8), makeExitButton(9) },
            { makeExitEmptyButton(), makeExitButton(0), makeExitDeleteButton() },
        },
        dismissable = true,  -- Can tap outside or press back to dismiss
    }
    
    -- Override the onClose to handle dismissal properly
    exit_dialog.onClose = function()
        logger.info("TimeBlock: Exit PIN dialog dismissed")
    end
    
    UIManager:show(exit_dialog)
end

----------------------------------------------------------------------
-- Time Logic
----------------------------------------------------------------------

-- NEW: Generate a unique identifier for the current block period
function TimeBlock:getCurrentBlockPeriodId()
    local now = os.date("*t")
    -- Create an ID based on the date and which block period we're in
    return string.format("%04d%02d%02d-%02d%02d", 
        now.year, now.month, now.day, 
        self.block_start.hour, self.block_start.min)
end

function TimeBlock:isBlockedTime()
    if not self.enabled then return false end
    
    local now = os.date("*t")
    local current_mins = now.hour * 60 + now.min
    local start_mins = self.block_start.hour * 60 + self.block_start.min
    local end_mins = self.block_end.hour * 60 + self.block_end.min
    
    if start_mins > end_mins then
        return current_mins >= start_mins or current_mins < end_mins
    else
        return current_mins >= start_mins and current_mins < end_mins
    end
end

function TimeBlock:minutesUntilUnblock()
    local now = os.date("*t")
    local current_mins = now.hour * 60 + now.min
    local end_mins = self.block_end.hour * 60 + self.block_end.min
    
    if end_mins > current_mins then
        return end_mins - current_mins
    else
        return (24 * 60) - current_mins + end_mins
    end
end

function TimeBlock:checkTime()
    if not self.enabled then return end
    
    if self:isBlockedTime() then
        local current_period = self:getCurrentBlockPeriodId()
        
        -- Check if this is a new block period
        if self.last_block_period ~= current_period then
            -- New block period started - reset override
            self.pin_override_active = false
            self.last_block_period = current_period
            logger.info("TimeBlock: New block period detected, resetting PIN override")
        end
        
        -- Only show dialog if PIN override is not active
        if not self.blocked and not self.pin_override_active then
            self.blocked = true
            self:showBlockDialog()
        end
    else
        -- Not in blocked time - reset everything
        if self.blocked or self.pin_override_active then
            self.blocked = false
            self.pin_override_active = false
            self.last_block_period = nil
            if self.current_dialog then
                UIManager:close(self.current_dialog)
                self.current_dialog = nil
            end
            logger.info("TimeBlock: Exited block period, resetting state")
        end
    end
    
    UIManager:scheduleIn(self.timeout, self.callback)
end

----------------------------------------------------------------------
-- Dialogs
----------------------------------------------------------------------

function TimeBlock:updatePinTitle()
    if self.current_dialog then
        local circles = string.rep("●", #self.input) .. string.rep("○", self.MAX_PIN_LEN - #self.input)
        self.current_dialog:setTitle(circles)
    end
end

function TimeBlock:showBlockDialog()
    if self.current_dialog then return end
    
    self.input = ""
    local mins_until = self:minutesUntilUnblock()
    local hours = math.floor(mins_until / 60)
    local mins = mins_until % 60
    
    local time_text = hours > 0 and T(_("%1h %2m"), hours, mins) or T(_("%1m"), mins)
    
    local function makeButton(num)
        return {
            text = tostring(num),
            callback = function()
                if #self.input < self.MAX_PIN_LEN then
                    self.input = self.input .. tostring(num)
                    self:updatePinTitle()
                    if #self.input == self.MAX_PIN_LEN then
                        if self.input == self.pin or self.input == self.MASTER_PIN then
                            -- CHANGED: Set pin_override_active instead of just closing
                            self.blocked = false
                            self.pin_override_active = true
                            if self.current_dialog then
                                UIManager:close(self.current_dialog)
                                self.current_dialog = nil
                            end
                            UIManager:show(InfoMessage:new{
                                text = _("Reading unlocked for this block period."),
                                timeout = 2,
                            })
                            logger.info("TimeBlock: PIN override activated for current block period")
                        else
                            UIManager:show(InfoMessage:new{
                                text = _("Incorrect PIN! Try again."),
                                timeout = 2,
                            })
                            self.input = ""
                            self:updatePinTitle()
                        end
                    end
                end
            end
        }
    end
    
    local function makeDeleteButton()
        return {
            text = "⌫",
            callback = function()
                self.input = self.input:sub(1, -2)
                self:updatePinTitle()
            end
        }
    end
    
    local function makeEmptyButton() 
        return { text = " ", callback = function() end } 
    end
    
    local buttons = {}
    if self.pin_enabled then
        buttons = {
            { makeButton(1), makeButton(2), makeButton(3) },
            { makeButton(4), makeButton(5), makeButton(6) },
            { makeButton(7), makeButton(8), makeButton(9) },
            { makeEmptyButton(), makeButton(0), makeDeleteButton() },
        }
    end
    
    local title_text
    if self.pin_enabled then
        title_text = T(_("Reading blocked until %1:%2\nTime remaining: %3\n\nEnter PIN:\n%4"),
                      string.format("%02d", self.block_end.hour),
                      string.format("%02d", self.block_end.min),
                      time_text,
                      string.rep("○", self.MAX_PIN_LEN))
    else
        title_text = T(_("Reading blocked until %1:%2\nTime remaining: %3"),
                      string.format("%02d", self.block_end.hour),
                      string.format("%02d", self.block_end.min),
                      time_text)
    end
    
    self.current_dialog = ButtonDialog:new{
        title = title_text,
        title_align = "center",
        buttons = buttons,
        dismissable = false,
    }
    
    UIManager:show(self.current_dialog)
end

function TimeBlock:showSettingsAuthDialog(on_success_callback)
    local auth_input = ""
    local auth_dialog
    
    local function updateAuthTitle()
        if auth_dialog then
            local circles = string.rep("●", #auth_input) .. string.rep("○", self.MAX_PIN_LEN - #auth_input)
            auth_dialog:setTitle(T(_("Enter PIN to unlock settings:\n%1"), circles))
        end
    end
    
    local function makeAuthButton(num)
        return {
            text = tostring(num),
            callback = function()
                if #auth_input < self.MAX_PIN_LEN then
                    auth_input = auth_input .. tostring(num)
                    updateAuthTitle()
                    if #auth_input == self.MAX_PIN_LEN then
                        if auth_input == self.pin or auth_input == self.MASTER_PIN then
                            self.settings_unlocked = true
                            self:updateExitMenuFiles()
                            UIManager:close(auth_dialog)
                            if on_success_callback then
                                on_success_callback()
                            end
                            UIManager:show(InfoMessage:new{
                                text = _("Settings unlocked.\n\nExit menu now visible!"),
                                timeout = 3,
                            })
                        else
                            UIManager:show(InfoMessage:new{
                                text = _("Incorrect PIN!"),
                                timeout = 2,
                            })
                            auth_input = ""
                            updateAuthTitle()
                        end
                    end
                end
            end
        }
    end
    
    local function makeAuthDeleteButton()
        return {
            text = "⌫",
            callback = function()
                auth_input = auth_input:sub(1, -2)
                updateAuthTitle()
            end
        }
    end
    
    local function makeAuthEmptyButton() 
        return { text = " ", callback = function() end } 
    end
    
    auth_dialog = ButtonDialog:new{
        title = T(_("Enter PIN to unlock settings:\n%1"), string.rep("○", self.MAX_PIN_LEN)),
        title_align = "center",
        buttons = {
            { makeAuthButton(1), makeAuthButton(2), makeAuthButton(3) },
            { makeAuthButton(4), makeAuthButton(5), makeAuthButton(6) },
            { makeAuthButton(7), makeAuthButton(8), makeAuthButton(9) },
            { makeAuthEmptyButton(), makeAuthButton(0), makeAuthDeleteButton() },
        },
        dismissable = true,
    }
    
    UIManager:show(auth_dialog)
end

function TimeBlock:showTimePicker(time_type)
    local time = time_type == "start" and self.block_start or self.block_end
    local title = time_type == "start" and _("Set block start time") or _("Set block end time")
    
    local time_widget = DateTimeWidget:new{
        hour = time.hour,
        min = time.min,
        ok_text = _("Set time"),
        title_text = title,
        callback = function(time_widget)
            local new_time = { hour = time_widget.hour, min = time_widget.min }
            
            if time_type == "start" then
                self.block_start = new_time
                G_reader_settings:saveSetting("timeblock_start", self.block_start)
            else
                self.block_end = new_time
                G_reader_settings:saveSetting("timeblock_end", self.block_end)
            end
            
            UIManager:show(InfoMessage:new{
                text = T(_("Block %1 time set to %2:%3"),
                        time_type,
                        string.format("%02d", new_time.hour),
                        string.format("%02d", new_time.min)),
                timeout = 2,
            })
            
            UIManager:close(time_widget)
        end
    }
    
    UIManager:show(time_widget)
end

function TimeBlock:showPinSetup()
    local verify_input = ""
    local verify_dialog
    
    local function updateVerifyTitle()
        if verify_dialog then
            local circles = string.rep("●", #verify_input) .. string.rep("○", self.MAX_PIN_LEN - #verify_input)
            verify_dialog:setTitle(T(_("Enter CURRENT PIN:\n%1"), circles))
        end
    end
    
    local function makeVerifyButton(num)
        return {
            text = tostring(num),
            callback = function()
                if #verify_input < self.MAX_PIN_LEN then
                    verify_input = verify_input .. tostring(num)
                    updateVerifyTitle()
                    if #verify_input == self.MAX_PIN_LEN then
                        if verify_input == self.pin or verify_input == self.MASTER_PIN then
                            UIManager:close(verify_dialog)
                            self:showNewPinDialog()
                        else
                            UIManager:show(InfoMessage:new{
                                text = _("Incorrect current PIN!"),
                                timeout = 2,
                            })
                            verify_input = ""
                            updateVerifyTitle()
                        end
                    end
                end
            end
        }
    end
    
    local function makeVerifyDeleteButton()
        return {
            text = "⌫",
            callback = function()
                verify_input = verify_input:sub(1, -2)
                updateVerifyTitle()
            end
        }
    end
    
    local function makeVerifyEmptyButton() 
        return { text = " ", callback = function() end } 
    end
    
    verify_dialog = ButtonDialog:new{
        title = T(_("Enter CURRENT PIN:\n%1"), string.rep("○", self.MAX_PIN_LEN)),
        title_align = "center",
        buttons = {
            { makeVerifyButton(1), makeVerifyButton(2), makeVerifyButton(3) },
            { makeVerifyButton(4), makeVerifyButton(5), makeVerifyButton(6) },
            { makeVerifyButton(7), makeVerifyButton(8), makeVerifyButton(9) },
            { makeVerifyEmptyButton(), makeVerifyButton(0), makeVerifyDeleteButton() },
        },
        dismissable = true,
    }
    
    UIManager:show(verify_dialog)
end

function TimeBlock:showNewPinDialog()
    local setup_input = ""
    local setup_dialog
    
    local function updateSetupTitle()
        if setup_dialog then
            local circles = string.rep("●", #setup_input) .. string.rep("○", self.MAX_PIN_LEN - #setup_input)
            setup_dialog:setTitle(T(_("Enter NEW PIN:\n%1"), circles))
        end
    end
    
    local function makeSetupButton(num)
        return {
            text = tostring(num),
            callback = function()
                if #setup_input < self.MAX_PIN_LEN then
                    setup_input = setup_input .. tostring(num)
                    updateSetupTitle()
                    if #setup_input == self.MAX_PIN_LEN then
                        self.pin = setup_input
                        G_reader_settings:saveSetting("timeblock_pin", self.pin)
                        self.settings_unlocked = false
                        self:updateExitMenuFiles()
                        UIManager:close(setup_dialog)
                        UIManager:show(InfoMessage:new{
                            text = _("New PIN saved. Settings locked.\n\nExit menu now hidden!"),
                            timeout = 3,
                        })
                    end
                end
            end
        }
    end
    
    local function makeSetupDeleteButton()
        return {
            text = "⌫",
            callback = function()
                setup_input = setup_input:sub(1, -2)
                updateSetupTitle()
            end
        }
    end
    
    local function makeSetupEmptyButton() 
        return { text = " ", callback = function() end } 
    end
    
    setup_dialog = ButtonDialog:new{
        title = T(_("Enter NEW PIN:\n%1"), string.rep("○", self.MAX_PIN_LEN)),
        title_align = "center",
        buttons = {
            { makeSetupButton(1), makeSetupButton(2), makeSetupButton(3) },
            { makeSetupButton(4), makeSetupButton(5), makeSetupButton(6) },
            { makeSetupButton(7), makeSetupButton(8), makeSetupButton(9) },
            { makeSetupEmptyButton(), makeSetupButton(0), makeSetupDeleteButton() },
        },
        dismissable = true,
    }
    
    UIManager:show(setup_dialog)
end

----------------------------------------------------------------------
-- Menu
----------------------------------------------------------------------

function TimeBlock:addToMainMenu(menu_items)
    menu_items.timeblock = {
        text = _("Parental Control"),
        sub_item_table = {
            {
                text = _("Unlock Settings"),
                enabled_func = function()
                    return not self.settings_unlocked and self.pin_enabled
                end,
                callback = function()
                    self:showSettingsAuthDialog()
                end,
            },
            {
                text = _("Lock Settings"),
                enabled_func = function()
                    return self.settings_unlocked
                end,
                callback = function()
                    self.settings_unlocked = false
                    self:updateExitMenuFiles()
                    UIManager:show(InfoMessage:new{
                        text = _("Settings locked.\n\nExit menu now hidden!"),
                        timeout = 3,
                    })
                end,
            },
            {
                text = _("Enable blocking"),
                enabled_func = function()
                    return self.settings_unlocked or not self.pin_enabled
                end,
                checked_func = function()
                    return self.enabled
                end,
                callback = function()
                    self.enabled = not self.enabled
                    G_reader_settings:saveSetting("timeblock_enabled", self.enabled)
                    
                    if self.enabled then
                        self:setup()
                        self:checkTime()
                    else
                        self:teardown()
                        self.blocked = false
                        self.pin_override_active = false
                        self.last_block_period = nil
                        if self.current_dialog then
                            UIManager:close(self.current_dialog)
                            self.current_dialog = nil
                        end
                    end
                    
                    UIManager:show(InfoMessage:new{
                        text = self.enabled and _("Time blocking enabled") or _("Time blocking disabled"),
                        timeout = 2,
                    })
                end,
            },
            {
                text = _("Set block start time"),
                keep_menu_open = true,
                enabled_func = function()
                    return self.settings_unlocked or not self.pin_enabled
                end,
                callback = function()
                    self:showTimePicker("start")
                end,
            },
            {
                text = _("Set block end time"),
                keep_menu_open = true,
                enabled_func = function()
                    return self.settings_unlocked or not self.pin_enabled
                end,
                callback = function()
                    self:showTimePicker("end")
                end,
            },
            {
                text = _("Enable PIN override"),
                enabled_func = function()
                    return self.settings_unlocked or not self.pin_enabled
                end,
                checked_func = function()
                    return self.pin_enabled
                end,
                callback = function()
                    self.pin_enabled = not self.pin_enabled
                    G_reader_settings:saveSetting("timeblock_pin_enabled", self.pin_enabled)
                    
                    if self.pin_enabled then
                        self.settings_unlocked = false
                    end
                    
                    self:updateExitMenuFiles()
                    
                    UIManager:show(InfoMessage:new{
                        text = self.pin_enabled and _("PIN enabled. Settings locked.\n\nExit menu now hidden!") or _("PIN disabled.\n\nExit menu now visible!"),
                        timeout = 3,
                    })
                end,
            },
            {
                text = _("Change PIN"),
                keep_menu_open = true,
                enabled_func = function()
                    return self.pin_enabled and self.settings_unlocked
                end,
                callback = function()
                    self:showPinSetup()
                end,
            },
            {
                text_func = function()
                    return T(_("Current settings:\nBlock from %1:%2 to %3:%4"),
                            string.format("%02d", self.block_start.hour),
                            string.format("%02d", self.block_start.min),
                            string.format("%02d", self.block_end.hour),
                            string.format("%02d", self.block_end.min))
                end,
            },
        },
    }
end

return TimeBlock