/* Sudoku for Illusion (Kindle) — ES5-only
 * Features: unique generator, difficulty, erase, undo/redo, check (conflict highlight), hint, new, reload.
 * Removed: notes & auto-notes (and all related logic).
 */

var SIZE = 9, BOX = 3;

/* State */
var solutionFull = null;   // full solved board kept hidden for hint/check
var puzzle = null;         // current puzzle board (0 = empty)
var given = null;          // boolean matrix: true = clue cell
var selected = null;       // {r,c}
var showErrors = false;    // Check mode
var difficulty = "medium"; // 'easy'|'medium'|'hard'
var undoStack = [];        // [{r,c,prevVal,newVal}]
var redoStack = [];

/* DOM helpers */
function $(id){ return document.getElementById(id); }
function setText(el, txt){ if (el) el.textContent = txt; }
function addClass(el, name){ if (!el) return; if ((' '+el.className+' ').indexOf(' '+name+' ')===-1) el.className = (el.className?el.className+' ':'')+name; }
function removeClass(el, name){ if (!el) return; el.className = (' '+el.className+' ').replace(' '+name+' ', ' ').replace(/^\s+|\s+$/g,''); }

/* Init */
document.addEventListener('DOMContentLoaded', function(){ init(); });

function init(){
  setText($('difficultyLabel'), capitalize(difficulty));
  newGame();
  window.addEventListener('resize', resizeLayout);
}

/* Difficulty */
function changeDifficulty(level){
  difficulty = level;
  setText($('difficultyLabel'), capitalize(level));
}

/* Button wrapper (prevents stuck focus highlight) */
function press(btn, action){
  if (btn && btn.blur) btn.blur();
  if (action === 'new') newGame();
  else if (action === 'reload'){ try{ window.location.reload(true); }catch(e){} }
}

/* New game: generate solution, carve puzzle with uniqueness */
function newGame(){
  setMessage("Generating...");
  solutionFull = generateSolvedBoard();       // full solution
  puzzle = cloneGrid(solutionFull);           // copy to puzzle
  given  = boolGrid(false);
  var removeCount = difficultyToRemoveCount(difficulty);
  removeCellsForUniquePuzzle(puzzle, removeCount); // ensures unique or best effort

  // mark clues
  var r,c;
  for (r=0;r<SIZE;r++){ for (c=0;c<SIZE;c++){ given[r][c] = (puzzle[r][c] !== 0); } }

  selected = null;
  undoStack = [];
  redoStack = [];
  showErrors = false;

  draw();
  setMessage("");
}

/* Draw grid */
function draw(){
  var t = $('grid');
  t.innerHTML = "";
  var r,c;
  for (r=0;r<SIZE;r++){
    var tr = document.createElement('tr');
    for (c=0;c<SIZE;c++){
      var td = document.createElement('td');
      td.className = 'cell ' + (given[r][c] ? 'clue' : 'user');
      var v = puzzle[r][c];
      if (v) setText(td, String(v));
      td.onclick = makeSelectHandler(r,c,td);
      tr.appendChild(td);
    }
    t.appendChild(tr);
  }

  if (selected){
    var idx = selected.r*SIZE + selected.c;
    var tds = t.getElementsByTagName('td');
    if (tds[idx]) addClass(tds[idx],'selected');
  }

  if (showErrors) highlightConflicts();
  resizeLayout();
}

function makeSelectHandler(r,c,td){
  return function(){
    if (given[r][c]) return; // can't select clues
    var tds = $('grid').getElementsByTagName('td');
    var i; for (i=0;i<tds.length;i++) removeClass(tds[i],'selected');
    selected = {r:r,c:c};
    addClass(td,'selected');
  };
}

/* Number pad */
function onNumberBtn(btn, n){
  if (btn && btn.blur) btn.blur();
  inputNumber(n);
}
function onEraseBtn(btn){
  if (btn && btn.blur) btn.blur();
  eraseCell();
}

function inputNumber(n){
  if (!selected){ setMessage("Select a cell first."); return; }
  var r=selected.r, c=selected.c;
  if (given[r][c]) return;
  var prev = puzzle[r][c];
  if (prev === n) return; // no change
  puzzle[r][c] = n;
  pushHistory(r,c,prev,n);
  redoStack = [];
  draw();

  if (isBoardComplete(puzzle)){
    if (gridsEqual(puzzle, solutionFull)) setMessage("Solved! 🎉");
    else setMessage("Complete but incorrect.");
  } else setMessage("");
}

function eraseCell(){
  if (!selected) return;
  var r=selected.r, c=selected.c;
  if (given[r][c]) return;
  var prev = puzzle[r][c];
  if (prev === 0) return;
  puzzle[r][c] = 0;
  pushHistory(r,c,prev,0);
  redoStack = [];
  draw();
}

/* Check & conflicts */
function checkBoard(btn){
  if (btn && btn.blur) btn.blur();
  showErrors = !showErrors;
  setMessage(showErrors ? "Check: On" : "Check: Off");
  draw();
}

function highlightConflicts(){
  var tds = $('grid').getElementsByTagName('td');
  var idx=0, r, c;
  for (r=0;r<SIZE;r++){
    for (c=0;c<SIZE;c++){
      removeClass(tds[idx],'error');
      var v = puzzle[r][c];
      if (v !== 0 && hasConflict(puzzle, r, c, v)) addClass(tds[idx],'error');
      idx++;
    }
  }
}

/* Hint (fill one random empty with the solution) */
function hint(btn){
  if (btn && btn.blur) btn.blur();
  var empties = emptyCells(puzzle);
  if (!empties.length){ setMessage("No hints available."); return; }
  var pick = empties[Math.floor(Math.random()*empties.length)];
  var r=pick.r, c=pick.c;
  var prev = puzzle[r][c]; // 0
  var val  = solutionFull[r][c];
  puzzle[r][c] = val;
  pushHistory(r,c,prev,val);
  redoStack = [];
  draw();
}

/* Undo/Redo */
function pushHistory(r,c,prevVal,newVal){
  undoStack.push({r:r,c:c,prevVal:prevVal,newVal:newVal});
  if (undoStack.length > 200) undoStack.shift();
}
function undoAction(btn){
  if (btn && btn.blur) btn.blur();
  if (!undoStack.length) return;
  var a = undoStack.pop();
  redoStack.push(a);
  puzzle[a.r][a.c] = a.prevVal;
  selected = {r:a.r,c:a.c};
  draw();
}
function redoAction(btn){
  if (btn && btn.blur) btn.blur();
  if (!redoStack.length) return;
  var a = redoStack.pop();
  // apply forward again
  var prev = puzzle[a.r][a.c];
  puzzle[a.r][a.c] = a.newVal;
  undoStack.push({r:a.r,c:a.c,prevVal:prev,newVal:a.newVal});
  selected = {r:a.r,c:a.c};
  draw();
}

/* Layout sizing */
function resizeLayout(){
  var screenW = window.innerWidth;
  var screenH = window.innerHeight;

  // Grid target size: majority of width, ~55% of height
  var gridSize = Math.min(Math.floor(screenW*0.95), Math.floor(screenH*0.57));
  var cellSize = Math.max(24, Math.floor(gridSize / SIZE));

  var tds = $('grid').getElementsByTagName('td');
  var i;
  for (i=0;i<tds.length;i++){
    tds[i].style.width = cellSize + "px";
    tds[i].style.height = cellSize + "px";
    tds[i].style.fontSize = Math.floor(cellSize*0.5) + "px";
    tds[i].style.lineHeight = cellSize + "px";
  }

  $('title').style.fontSize   = Math.floor(gridSize/12) + "px";
  $('scoreBox').style.fontSize= Math.floor(gridSize/20) + "px";
  $('message').style.fontSize = Math.floor(gridSize/26) + "px";

  // number pad buttons sizing
  var rows = $('numberPad').getElementsByClassName('np-row');
  for (i=0;i<rows.length;i++){
    var btns = rows[i].getElementsByTagName('button');
    var j;
    for (j=0;j<btns.length;j++){
      btns[j].style.minWidth = Math.floor(cellSize*1.2) + "px";
      btns[j].style.height   = Math.floor(cellSize*0.9) + "px";
      btns[j].style.fontSize = Math.floor(cellSize*0.36) + "px";
    }
  }

  var acts = $('actions').getElementsByTagName('button');
  for (i=0;i<acts.length;i++){
    acts[i].style.minWidth = Math.floor(cellSize*2.0) + "px";
    acts[i].style.height   = Math.floor(cellSize*0.95) + "px";
    acts[i].style.fontSize = Math.floor(cellSize*0.38) + "px";
  }
}

/* Messages */
function setMessage(m){ setText($('message'), m||""); }

/* ===== Generator & Solver (ES5) ===== */

/* Utilities */
function capitalize(s){ s=String(s); return s.charAt(0).toUpperCase()+s.slice(1); }
function cloneGrid(g){
  var r,c,out=[]; for (r=0;r<SIZE;r++){ out[r]=[]; for (c=0;c<SIZE;c++){ out[r][c]=g[r][c]; } }
  return out;
}
function gridsEqual(a,b){
  var r,c; for (r=0;r<SIZE;r++){ for (c=0;c<SIZE;c++){ if (a[r][c]!==b[r][c]) return false; } }
  return true;
}
function boolGrid(val){
  var r,c,g=[]; for (r=0;r<SIZE;r++){ g[r]=[]; for (c=0;c<SIZE;c++){ g[r][c]=val; } }
  return g;
}
function emptyCells(b){
  var r,c,arr=[]; for (r=0;r<SIZE;r++){ for (c=0;c<SIZE;c++){ if (b[r][c]===0) arr.push({r:r,c:c}); } }
  return arr;
}

/* Sudoku rules helpers */
function isSafe(b,row,col,val){
  var i,j, br=Math.floor(row/BOX)*BOX, bc=Math.floor(col/BOX)*BOX;
  for (i=0;i<SIZE;i++){ if (b[i][col]===val) return false; }
  for (j=0;j<SIZE;j++){ if (b[row][j]===val) return false; }
  for (i=0;i<BOX;i++){ for (j=0;j<BOX;j++){ if (b[br+i][bc+j]===val) return false; } }
  return true;
}
function hasConflict(b,row,col,val){
  var i,j, br=Math.floor(row/BOX)*BOX, bc=Math.floor(col/BOX)*BOX;
  for (i=0;i<SIZE;i++){ if (i!==row && b[i][col]===val) return true; }
  for (j=0;j<SIZE;j++){ if (j!==col && b[row][j]===val) return true; }
  for (i=0;i<BOX;i++){ for (j=0;j<BOX;j++){ var R=br+i,C=bc+j; if ((R!==row||C!==col) && b[R][C]===val) return true; } }
  return false;
}
function isBoardComplete(b){
  var r,c; for (r=0;r<SIZE;r++){ for (c=0;c<SIZE;c++){ if (b[r][c]===0) return false; } }
  return true;
}

/* Difficulty mapping: cells to remove */
function difficultyToRemoveCount(level){
  if (level==='easy') return 40;   // leaves ~41 clues
  if (level==='hard') return 54;   // leaves ~27 clues
  return 48;                       // medium ~33 clues
}

/* Generate a full valid solution via randomized backtracking */
function generateSolvedBoard(){
  var r,c,board=[]; for (r=0;r<SIZE;r++){ board[r]=[]; for (c=0;c<SIZE;c++){ board[r][c]=0; } }
  solveFill(board,0,0);
  return board;
}
function solveFill(b,row,col){
  if (row===SIZE) return true;
  var nextRow = (col===SIZE-1) ? row+1 : row;
  var nextCol = (col===SIZE-1) ? 0 : col+1;

  var digits=[1,2,3,4,5,6,7,8,9]; shuffle(digits);
  var i,d;
  for (i=0;i<digits.length;i++){
    d=digits[i];
    if (isSafe(b,row,col,d)){
      b[row][col]=d;
      if (solveFill(b,nextRow,nextCol)) return true;
      b[row][col]=0;
    }
  }
  return false;
}
function shuffle(a){
  var i=a.length-1; for (; i>0; i--){ var j=Math.floor(Math.random()*(i+1)), t=a[i]; a[i]=a[j]; a[j]=t; }
}

/* Remove cells while keeping uniqueness (countSolutions==1 when possible) */
function removeCellsForUniquePuzzle(b, removeCount){
  var cells=[], r,c;
  for (r=0;r<SIZE;r++){ for (c=0;c<SIZE;c++){ cells.push({r:r,c:c}); } }
  shuffle(cells);

  var removed=0, tries=0, maxTries=5000, idx=0;
  while (removed<removeCount && tries<maxTries && idx<cells.length){
    var pos = cells[idx++]; tries++;
    if (b[pos.r][pos.c]===0) continue;
    var backup=b[pos.r][pos.c];
    b[pos.r][pos.c]=0;

    var work=cloneGrid(b);
    var cnt = countSolutions(work, 2); // early stop at 2
    if (cnt!==1){
      b[pos.r][pos.c]=backup; // not unique, revert
    } else {
      removed++;
    }
  }
  // If we fall short, accept best-effort to keep generation time reasonable on Kindle
}

/* Count solutions up to 'limit' using MRV heuristic */
function countSolutions(b, limit){
  var count=0;

  function collectCandidates(row,col){
    var used={}, r,c, br=Math.floor(row/BOX)*BOX, bc=Math.floor(col/BOX)*BOX, out=[], v;
    for (r=0;r<SIZE;r++){ v=b[r][col]; if (v) used[v]=true; }
    for (c=0;c<SIZE;c++){ v=b[row][c]; if (v) used[v]=true; }
    for (r=0;r<BOX;r++){ for (c=0;c<BOX;c++){ v=b[br+r][bc+c]; if (v) used[v]=true; } }
    for (v=1; v<=9; v++){ if (!used[v]) out.push(v); }
    return out;
  }

  function findBestCell(){
    var min=10, best=null, r,c, cands;
    for (r=0;r<SIZE;r++){
      for (c=0;c<SIZE;c++){
        if (b[r][c]===0){
          cands = collectCandidates(r,c);
          if (cands.length<min){ min=cands.length; best={r:r,c:c,cands:cands}; if (min===0) return best; }
        }
      }
    }
    return best;
  }

  function backtrack(){
    if (count>=limit) return;
    var cell = findBestCell();
    if (!cell){ count++; return; }
    var i, d;
    for (i=0;i<cell.cands.length;i++){
      d=cell.cands[i];
      b[cell.r][cell.c]=d;
      backtrack();
      b[cell.r][cell.c]=0;
      if (count>=limit) return;
    }
  }

  backtrack();
  return count;
}
