/* Snake for Illusion — ES5 only, no flexbox, dynamic sizing
   by kindlemodshelfguy */

var COLS = 18;  // grid width
var ROWS = 13;  // grid height
var TICK = 300; // ms per move (eInk-friendly)

/* Directions */
var DIR = { left:{x:-1,y:0}, right:{x:1,y:0}, up:{x:0,y:-1}, down:{x:0,y:1} };

/* State */
var snake, dir, nextDir, food, score, timer, alive, paused;
var turnedThisTick = false; // debounces double-turn glitches

/* Utils */
function $(id){ return document.getElementById(id); }
function setText(id, s){ var el=$(id); if(el){ el.innerHTML = s; } }
function blurBtn(b){ try{ if(b && b.blur){ b.blur(); } }catch(e){} }
function setMsg(s){ setText("message", s || ""); }

/* Build the grid table once */
function buildGrid(){
  var t = $("grid");
  if(!t) return;
  t.innerHTML = "";
  var r,c,tr,td;
  for(r=0;r<ROWS;r++){
    tr = document.createElement("tr");
    for(c=0;c<COLS;c++){
      td = document.createElement("td");
      tr.appendChild(td);
    }
    t.appendChild(tr);
  }
}

/* New game setup */
function newGame(){
  score = 0; setText("score", "0");
  alive = true; paused = false; turnedThisTick = false;

  snake = [];
  var sx = Math.floor(COLS/2), sy = Math.floor(ROWS/2);
  snake.push({x:sx,   y:sy});
  snake.push({x:sx-1, y:sy});
  snake.push({x:sx-2, y:sy});

  dir = DIR.right; nextDir = dir;

  placeFood();
  draw();

  if (timer) { clearInterval(timer); }
  timer = setInterval(tick, TICK);
  setMsg("");
  resizeLayout();
}

/* Place food in an empty cell */
function placeFood(){
  var empties = [];
  var x,y,i,occupied;
  for(y=0;y<ROWS;y++){
    for(x=0;x<COLS;x++){
      occupied = false;
      for(i=0;i<snake.length;i++){
        if(snake[i].x===x && snake[i].y===y){ occupied = true; break; }
      }
      if(!occupied){ empties.push({x:x,y:y}); }
    }
  }
  if(empties.length === 0){
    // filled board — win
    alive = false; if (timer) clearInterval(timer);
    setMsg("You win! Score: " + score);
    return;
  }
  food = empties[Math.floor(Math.random()*empties.length)];
}

/* Main loop */
function tick(){
  if(!alive || paused){ return; }
  // Allow exactly one turn per tick window:
  turnedThisTick = false;

  dir = nextDir;
  var head = snake[0];
  var nx = head.x + dir.x;
  var ny = head.y + dir.y;

  // walls
  if(nx<0 || nx>=COLS || ny<0 || ny>=ROWS){ return gameOver(); }

  // self collision
  var i;
  for(i=0;i<snake.length;i++){
    if(snake[i].x===nx && snake[i].y===ny){ return gameOver(); }
  }

  // step forward
  snake.unshift({x:nx, y:ny});

  // food?
  if(food && nx===food.x && ny===food.y){
    score += 1; setText("score", String(score));
    placeFood();
  } else {
    snake.pop(); // move tail
  }

  draw();
}

/* Render */
function draw(){
  var grid = $("grid");
  if(!grid) return;
  var cells = grid.getElementsByTagName("td");
  var total = COLS*ROWS, i;
  // clear
  for(i=0;i<total;i++){
    if(cells[i]) cells[i].className = "";
  }

  // food
  if(food){
    var fi = food.y*COLS + food.x;
    if(cells[fi]){ cells[fi].className = "cell-food"; }
  }

  // snake
  for(i=0;i<snake.length;i++){
    var s = snake[i];
    var idx = s.y*COLS + s.x;
    if(cells[idx]){
      cells[idx].className = (i===0) ? "cell-head" : "cell-snake";
    }
  }

  resizeLayout(); // keep sizing synced to viewport
}

/* End game */
function gameOver(){
  alive = false; if (timer) clearInterval(timer);
  setMsg("Game Over — Score: " + score);
}

/* Button actions */
function press(btn, action){
  // ensure we only blur if btn is provided
  try{ blurBtn(btn); }catch(e){}

  if(action === "new"){ newGame(); return; }
  if(action === "reload"){ reloadApp(); return; }
  if(action === "pause"){ if(alive){ paused = !paused; setMsg(paused?"Paused":""); } return; }

  if(!alive || paused){ return; }

  // Direction changes: disallow 180º reversals and only one change per tick
  if(turnedThisTick){ return; }

  // compare by coordinates (objects can't be compared with !== reliably across contexts)
  if(action==="left"  && !(dir.x===1 && dir.y===0)){ nextDir = DIR.left;  turnedThisTick=true; return; }
  if(action==="right" && !(dir.x===-1&& dir.y===0)){ nextDir = DIR.right; turnedThisTick=true; return; }
  if(action==="up"    && !(dir.x===0 && dir.y===1)){ nextDir = DIR.up;    turnedThisTick=true; return; }
  if(action==="down"  && !(dir.x===0 && dir.y===-1)){ nextDir = DIR.down;  turnedThisTick=true; return; }
}

/* Explicit reload function — only called from the reload button */
function reloadApp(){
  try { window.location.reload(); } catch(e) { /* swallow on old webkit */ }
}

/* Keyboard controls for desktop testing (and hardware keys if mapped) */
function onKey(e){
  var k = e.keyCode || e.which;
  if(k===37||k===38||k===39||k===40||k===32){ if(e.preventDefault){ e.preventDefault(); } }

  if(k===78){ newGame(); return; }        // N = new
  if(k===82){ try{ window.location.reload(); }catch(x){} return; } // R = restart
  if(k===32){ press(null,"pause"); return; } // Space = pause

  if(k===37||k===65){ press(null,"left");  return; }  // ← or A
  if(k===39||k===68){ press(null,"right"); return; }  // → or D
  if(k===38||k===87){ press(null,"up");    return; }  // ↑ or W
  if(k===40||k===83){ press(null,"down");  return; }  // ↓ or S
}

/* Dynamic sizing to fit without scrolling */
function resizeLayout(){
  var w = window.innerWidth || document.documentElement.clientWidth || 600;
  var h = window.innerHeight || document.documentElement.clientHeight || 800;

  // Measure non-grid vertical space (title, subtitle, score, controls, actions, message)
  var top    = document.getElementById("top");
  var ctrl   = document.getElementById("ctrlTable");
  var acts   = document.getElementById("actionsTable");
  var msg    = document.getElementById("message");

  var topH  = top  ? top.offsetHeight  : 0;
  var ctrlH = ctrl ? ctrl.offsetHeight : 0;
  var actH  = acts ? acts.offsetHeight : 0;
  var msgH  = msg  ? msg.offsetHeight  : 0;

  // Margins/padding estimate
  var extra = 16;

  var availableH = h - (topH + ctrlH + actH + msgH + extra);
  if(availableH < 80){ availableH = 80; }

  var gridMaxW = Math.floor(w * 0.94);
  var gridMaxH = Math.floor(availableH);

  var cellW = Math.floor(gridMaxW / COLS);
  var cellH = Math.floor(gridMaxH / ROWS);
  var cell  = Math.max(14, Math.min(cellW, cellH)); // readable minimum

  // Scale font sizes a bit with cell
  var tSize = Math.max(18, Math.floor(cell * 1.0));
  var sSize = Math.max(12, Math.floor(cell * 0.8));
  var mSize = Math.max(12, Math.floor(cell * 0.7));

  var titleEl = document.getElementById("title");
  if (titleEl) titleEl.style.fontSize = tSize + "px";
  var scoreEl = document.getElementById("scoreBox");
  if (scoreEl) scoreEl.style.fontSize = sSize + "px";
  var msgEl = document.getElementById("message");
  if (msgEl) msgEl.style.fontSize = mSize + "px";

  // Apply to grid cells
  var grid = document.getElementById("grid");
  if (!grid) return;
  var cells = grid.getElementsByTagName("td");
  var i;
  for(i=0;i<cells.length;i++){
    cells[i].style.width  = cell + "px";
    cells[i].style.height = cell + "px";
    cells[i].style.lineHeight = cell + "px";
  }

  // Resize buttons
  var ctrlButtons = document.getElementsByClassName("ctrl");
  var ctrlW = Math.max(64, Math.floor(cell * 3.0));
  var ctrlH2 = Math.max(32, Math.floor(cell * 1.6));
  var ctrlFont = Math.max(16, Math.floor(cell * 0.9));
  for(i=0;i<ctrlButtons.length;i++){
    ctrlButtons[i].style.width = ctrlW + "px";
    ctrlButtons[i].style.height = ctrlH2 + "px";
    ctrlButtons[i].style.fontSize = ctrlFont + "px";
  }

  var actionButtons = document.getElementsByClassName("action");
  var aW = Math.max(110, Math.floor(cell * 3.5));
  var aH = Math.max(32, Math.floor(cell * 1.6));
  var aF = Math.max(14, Math.floor(cell * 0.85));
  for(i=0;i<actionButtons.length;i++){
    actionButtons[i].style.width = aW + "px";
    actionButtons[i].style.height = aH + "px";
    actionButtons[i].style.fontSize = aF + "px";
  }
}

/* Init */
document.addEventListener("DOMContentLoaded", function(){
  buildGrid();
  newGame();
  window.addEventListener("resize", resizeLayout);
  document.addEventListener("keydown", onKey);
});
