#include <curses.h>
#include <stdlib.h>
#include <time.h>
#include <stdbool.h>
#include <locale.h>
#include <sys/time.h>

#define BOARD_WIDTH 10
#define BOARD_HEIGHT 20
#define TETROMINO_COUNT 7

typedef enum {I, O, T, L, J, S, Z} TetrominoType;

// Tetromino shapes [type][rotation][y][x]
const int SHAPES[TETROMINO_COUNT][4][4][4] = {
    [I] = {{{0,0,0,0}, {1,1,1,1}, {0,0,0,0}, {0,0,0,0}},
           {{0,0,1,0}, {0,0,1,0}, {0,0,1,0}, {0,0,1,0}},
           {{0,0,0,0}, {0,0,0,0}, {1,1,1,1}, {0,0,0,0}},
           {{0,1,0,0}, {0,1,0,0}, {0,1,0,0}, {0,1,0,0}}},
    [O] = {{{0,1,1,0}, {0,1,1,0}, {0,0,0,0}, {0,0,0,0}},
           {{0,1,1,0}, {0,1,1,0}, {0,0,0,0}, {0,0,0,0}},
           {{0,1,1,0}, {0,1,1,0}, {0,0,0,0}, {0,0,0,0}},
           {{0,1,1,0}, {0,1,1,0}, {0,0,0,0}, {0,0,0,0}}},
    [T] = {{{0,1,0,0}, {1,1,1,0}, {0,0,0,0}, {0,0,0,0}},
           {{0,1,0,0}, {0,1,1,0}, {0,1,0,0}, {0,0,0,0}},
           {{0,0,0,0}, {1,1,1,0}, {0,1,0,0}, {0,0,0,0}},
           {{0,1,0,0}, {1,1,0,0}, {0,1,0,0}, {0,0,0,0}}},
    [L] = {{{0,0,1,0}, {1,1,1,0}, {0,0,0,0}, {0,0,0,0}},
           {{0,1,0,0}, {0,1,0,0}, {0,1,1,0}, {0,0,0,0}},
           {{0,0,0,0}, {1,1,1,0}, {1,0,0,0}, {0,0,0,0}},
           {{1,1,0,0}, {0,1,0,0}, {0,1,0,0}, {0,0,0,0}}},
    [J] = {{{1,0,0,0}, {1,1,1,0}, {0,0,0,0}, {0,0,0,0}},
           {{0,1,1,0}, {0,1,0,0}, {0,1,0,0}, {0,0,0,0}},
           {{0,0,0,0}, {1,1,1,0}, {0,0,1,0}, {0,0,0,0}},
           {{0,1,0,0}, {0,1,0,0}, {1,1,0,0}, {0,0,0,0}}},
    [S] = {{{0,1,1,0}, {1,1,0,0}, {0,0,0,0}, {0,0,0,0}},
           {{0,1,0,0}, {0,1,1,0}, {0,0,1,0}, {0,0,0,0}},
           {{0,0,0,0}, {0,1,1,0}, {1,1,0,0}, {0,0,0,0}},
           {{1,0,0,0}, {1,1,0,0}, {0,1,0,0}, {0,0,0,0}}},
    [Z] = {{{1,1,0,0}, {0,1,1,0}, {0,0,0,0}, {0,0,0,0}},
           {{0,0,1,0}, {0,1,1,0}, {0,1,0,0}, {0,0,0,0}},
           {{0,0,0,0}, {1,1,0,0}, {0,1,1,0}, {0,0,0,0}},
           {{0,1,0,0}, {1,1,0,0}, {1,0,0,0}, {0,0,0,0}}}
};

typedef struct {
    TetrominoType type;
    int rotation, x, y;
} Tetromino;

int board[BOARD_HEIGHT][BOARD_WIDTH] = {0};
int LEFT_BORDER=0;
Tetromino current, next_piece;
int score = 0, level = 1, lines = 0;
bool game_over = false;

void init_game() {
    for (int y = 0; y < BOARD_HEIGHT; y++)
        for (int x = 0; x < BOARD_WIDTH; x++)
            board[y][x] = 0;
    current.type = rand() % TETROMINO_COUNT;
    current.rotation = 0;
    current.x = BOARD_WIDTH / 2 - 2;
    current.y = 0;
    next_piece.type = rand() % TETROMINO_COUNT;
    next_piece.rotation = 0;
    score = lines = 0;
    level = 1;
    game_over = false;
}

int check_collision(Tetromino *t) {
    for (int y = 0; y < 4; y++) {
        for (int x = 0; x < 4; x++) {
            if (SHAPES[t->type][t->rotation][y][x]) {
                int nx = t->x + x, ny = t->y + y;
                if (nx < 0 || nx >= BOARD_WIDTH || ny >= BOARD_HEIGHT) return 1;
                if (ny >= 0 && board[ny][nx]) return 1;
            }
        }
    }
    return 0;
}

void merge_tetromino(Tetromino *t) {
    for (int y = 0; y < 4; y++) {
        for (int x = 0; x < 4; x++) {
            if (SHAPES[t->type][t->rotation][y][x]) {
                int nx = t->x + x, ny = t->y + y;
                if (ny >= 0 && nx >= 0 && nx < BOARD_WIDTH)
                    board[ny][nx] = 1;
            }
        }
    }
}

int clear_lines() {
    int cleared = 0;
    for (int y = BOARD_HEIGHT -1; y >= 0; y--) {
        int full = 1;
        for (int x = 0; x < BOARD_WIDTH; x++)
            if (!board[y][x]) { full = 0; break; }
        if (full) {
            for (int yy = y; yy > 0; yy--)
                for (int x = 0; x < BOARD_WIDTH; x++)
                    board[yy][x] = board[yy-1][x];
            for (int x = 0; x < BOARD_WIDTH; x++) board[0][x] = 0;
            y++;
            cleared++;
        }
    }
    return cleared;
}

void rotate(int dir) {
    Tetromino tmp = current;
    tmp.rotation = (current.rotation + dir +4) %4;
    if (!check_collision(&tmp)) current = tmp;
}

void draw_board_borders() {
    // Horizontal borders
    for(int x = LEFT_BORDER+2; x < LEFT_BORDER+BOARD_WIDTH*2 + 2; x++) {
        //mvaddch(0, x, '-');
        mvaddch(BOARD_HEIGHT+1, x, '-');
    }
    
    // Vertical borders
    for(int y = 1; y <= BOARD_HEIGHT; y++) {
        mvaddch(y, LEFT_BORDER+1, '|');
        mvaddch(y, LEFT_BORDER+BOARD_WIDTH*2 + 2, '|');
    }
    
    // Corners
    mvaddch(0, LEFT_BORDER+1, '+');
    mvaddch(0, LEFT_BORDER+BOARD_WIDTH*2 + 2, '+');
    mvaddch(BOARD_HEIGHT+1, LEFT_BORDER+1, '+');
    mvaddch(BOARD_HEIGHT+1, LEFT_BORDER+BOARD_WIDTH*2 + 2, '+');
}

void draw() {
    erase();  // More efficient than clear() for partial updates
    draw_board_borders();
    
    // Draw board cells
    for (int y = 0; y < BOARD_HEIGHT; y++) {
        for (int x = 0; x < BOARD_WIDTH; x++) {
            mvaddstr(y+1, LEFT_BORDER+(x+1)*2, board[y][x] ? "[]" : "  ");
        }
    }

    // Draw current piece
    for (int y = 0; y < 4; y++) {
        for (int x = 0; x < 4; x++) {
            if (SHAPES[current.type][current.rotation][y][x]) {
                mvaddstr(current.y + y + 1, LEFT_BORDER+(current.x + x + 1)*2, "[]");
            }
        }
    }

    // Info panel
    int px = LEFT_BORDER - 6;
    mvprintw(1 , px, "#  #");
    mvprintw(2 , px, "# # #");
    mvprintw(3 , px, " #  #");
    mvprintw(5 , px, "#####");
    mvprintw(7 , px, "#####");
    mvprintw(8 , px, "# #");
    mvprintw(9 , px, " # ##");
    mvprintw(11, px, "#");
    mvprintw(12, px, "#####");
    mvprintw(13, px, "#");
    mvprintw(15, px, "# # #");
    mvprintw(16, px, "# # #");
    mvprintw(17, px, "#####");
    mvprintw(19, px, "#");
    mvprintw(20, px, "#####");
    mvprintw(21, px, "#");
    px = LEFT_BORDER + (BOARD_WIDTH + 1)*2 + 6;
    mvprintw(8, px, "Next:");
    for (int y = 0; y < 4; y++) {
        for (int x = 0; x < 4; x++) {
            mvaddstr(y+10, px + x*2, SHAPES[next_piece.type][0][y][x] ? "[]" : "  ");
        }
    }
    mvprintw(15, px, "Score: %d", score);
    mvprintw(16, px, "Level: %d", level);
    mvprintw(17, px, "Lines: %d", lines);

    refresh();
}

void game_loop() {
    struct timeval tv;
    gettimeofday(&tv, NULL);
    long last_fall = tv.tv_sec *1000 + tv.tv_usec /1000;
    int fall_delay = 1000;
    while (!game_over) {
        gettimeofday(&tv, NULL);
        long now = tv.tv_sec *1000 + tv.tv_usec /1000;
        int ch = getch();
        switch(ch) {
            case KEY_LEFT:
            case 'j': 
                current.x--; if(check_collision(&current)) current.x++; break;
            case KEY_RIGHT: 
            case 'l': 
                current.x++; if(check_collision(&current)) current.x--; break;
            case KEY_DOWN: 
                current.y++; if(check_collision(&current)) current.y--; else last_fall = now; break;
            case KEY_UP:
            case 'k': 
                rotate(1); break;
            case ' ': while(!check_collision(&current)) current.y++; current.y--; break;
            case 'q': game_over = true; return;
        }
        if (now - last_fall >= fall_delay) {
            current.y++;
            if (check_collision(&current)) {
                current.y--;
                merge_tetromino(&current);
                int cleared = clear_lines();
                lines += cleared;
                score += cleared ==1 ? 100*level : cleared ==2 ? 300*level :
                         cleared ==3 ? 500*level : cleared ==4 ? 800*level : 0;
                if (lines >= level*10) level++;
                fall_delay = 1000 - (level-1)*100;
                if (fall_delay < 100) fall_delay = 100;
                current = next_piece;
                current.x = BOARD_WIDTH/2 -2;
                current.y =0;
                next_piece.type = rand()%TETROMINO_COUNT;
                next_piece.rotation =0;
                if (check_collision(&current)) game_over = true;
            }
            last_fall = now;
        }
        draw();
    }
}

int main() {
    setlocale(LC_ALL, "");
    initscr(); cbreak(); noecho(); keypad(stdscr, TRUE); timeout(0);
    srand(time(NULL));
    curs_set(0);
    bool replay = true;
    LEFT_BORDER = (COLS-20)/2;
    if(LEFT_BORDER<0)LEFT_BORDER=0;
    while(replay) {
        init_game();
        game_loop();
        clear();
        mvprintw(10, COLS/2-5, "Game Over!");
        mvprintw(11, COLS/2-7, "Score: %d", score);
        mvprintw(12, COLS/2-10, "Play again? (Y/N)");
        refresh();
        timeout(-1);
        int ch = getch();
        replay = (ch == 'y' || ch == 'Y');
    }
    endwin();
    return 0;
}
