/* Hangman for Illusion — Kindle-safe (ES5 only; no flex/grid/transforms) */

var MAX_WRONG = 6;

/* Expanded word bank */
var WORDS = [
  "hangman","puzzle","reader","bookshelf","chapter","library","display","battery","wireless","internet",
  "browser","canvas","widget","javascript","stylesheet","bookmark","airplane","bicycle","chocolate","mountain",
  "ocean","desert","island","penguin","galaxy","asteroid","gravity","satellite","rocket","orbit",
  "eclipse","lunar","solar","syntax","variable","function","object","array","closure","callback",
  "storage","compiler","coffee","sandwich","avocado","banana","strawberry","blueberry","pancake","waffle",
  "noodle","adventure","mystery","fantasy","history","science","fiction","romance","classic","poetry",
  "winter","spring","summer","autumn","thunder","blizzard","rainstorm","breeze","sunshine","keyboard",
  "monitor","speaker","headphone","charger","adapter","bluetooth","touch","ink","notebook","journal",
  "calendar","reminder","message","session","progress","project","snippet","pirate","castle","forest",
  "descent","voyage","compass","lantern","harbor","bridge","river","waterfall","valley","canyon",
  "cliff","plateau","bay","beach","coast","shore","reef","lagoon","peninsula","plain",
  "prairie","meadow","hill","peak","summit","ridge","slope","cave","grotto","cabin",
  "village","town","city","metropolis","market","harvest","farm","orchard","garden","greenhouse",
  "vineyard","field","path","trail","journey","expedition","explore","discover","ancient","modern",
  "urban","rural","remote","hidden","secret","treasure","map","route","signal","beacon",
  "lighthouse","dock","wharf","port","terminal","station","platform","ticket","passport","customs",
  "border","companion","traveler","guide","pilot","captain","engineer","doctor","teacher","student",
  "artist","writer","poet","actor","singer","musician","painter","sculptor","dancer","gardener",
  "chef","baker","barista","butcher","farmer","brewer","winemaker","tailor","shoemaker","carpenter",
  "builder","architect","designer","photographer","chemist","physicist","biologist","astronomer","mathematician","philosopher",
  "historian","librarian","curator","collector","researcher","inventor","engine","motor","gear","wheel",
  "sail","rudder","anchor","cockpit","lamp","torch","candle","flame","ember","ash",
  "smoke","flag","banner","shield","sword","armor","helmet","scroll","manuscript","diary",
  "letter","note","voice","whisper","shout","sound","silence","echo","melody","rhythm",
  "beat","chord","harmony","tune","song","anthem","ballad","serenade","opera","trio",
  "quartet","orchestra","concert","festival","stage","screen","film","movie","drama","comedy",
  "tragedy","scene","script","dialogue","plot","novel","series","episode","season","character",
  "hero","villain","sidekick","mentor","apprentice","enemy","ally","friend","family","mother",
  "father","sister","brother","child","parent","grandparent","uncle","aunt","cousin","neighbor",
  "colleague","partner","spouse","husband","wife","lover","affection","friendship","trust","honor",
  "duty","courage","bravery","valor","glory","victory","defeat","battle","war","peace",
  "truce","treaty","siege","revolt","rebellion","uprising","riot","protest","speech","debate",
  "council","assembly","voting","election","campaign","leader","ruler","king","queen","prince",
  "princess","duke","duchess","knight","lord","lady","peasant","artisan","merchant","trader",
  "shop","boutique","marketplace","mall","plaza","square","avenue","street","alley","lane",
  "boulevard","highway","pathway","crossing","intersection","tunnel","ferry","rail","track","station",
  "harbor","pier","yacht","vessel","boat","ship","submarine","cruise","camp","tent",
  "shelter","hut","cottage","mansion","palace","villa","manor","estate","farmhouse","barn",
  "stable","kitchen","pantry","cellar","attic","porch","balcony","terrace","yard","lawn",
  "courtyard","patio","driveway","garage","workshop","studio","office","desk","chair","table",
  "shelf","cabinet","drawer","closet","mirror","window","door","gate","fence","wall",
  "roof","chimney","pipe","vent","stair","ladder","ramp","elevator","corridor","hall",
  "lobby","foyer","vestibule","auditorium","museum","gallery","archive","collection","exhibit","showcase",
  "catalog","index","database","record","file","folder","document","report","paper","article",
  "essay","review","critique","analysis","summary","abstract","glossary","appendix","preface","introduction",
  "conclusion","epilogue","prologue","section","paragraph","sentence","phrase","word","letter","alphabet"
];

/* State */
var secret = "", revealed = [], guessed = {}, wrongCount = 0;

/* Helpers */
function $(id){ try{ return document.getElementById(id); }catch(e){ return null; } }
function setText(id, txt){ var el=$(id); if(!el) return; try{ el.textContent = txt; }catch(e){ el.innerHTML = txt; } }
function setMsg(s){ setText("message", s || ""); }
function clamp(v,min,max){ return v<min?min:(v>max?max:v); }

/* ==== BUILD ==== */
function buildKeyboard(){
  var rows = ["ABCDEFGHIJK","LMNOPQRSTU","VWXYZ"];
  var t = $("keyboard"); if(!t) return;
  t.innerHTML = "";
  for(var r=0;r<rows.length;r++){
    var tr = document.createElement("tr");
    for(var i=0;i<rows[r].length;i++){
      var ch = rows[r].charAt(i);
      var td = document.createElement("td");
      var btn = document.createElement("button");
      btn.type = "button";
      btn.className = "letter";
      btn.id = "key_" + ch;
      btn.innerHTML = ch;
      (function(letter){
        btn.onclick = function(){ onLetterPress(letter.toLowerCase()); };
      })(ch);
      td.appendChild(btn);
      tr.appendChild(td);
    }
    t.appendChild(tr);
  }
}

function drawGallows(){
  var g = $("gallows"); if(!g) return;
  g.innerHTML = "";

  function mk(id){
    var d = document.createElement("div");
    d.className = "gpart"; d.id = id; return d;
  }

  // Structure is always visible
  g.appendChild(mk("g-base"));
  g.appendChild(mk("g-pole"));
  g.appendChild(mk("g-beam"));
  g.appendChild(mk("g-rope"));

  // Parts by error count
  if(wrongCount >= 1) g.appendChild(mk("g-head"));
  if(wrongCount >= 2) g.appendChild(mk("g-body"));
  if(wrongCount >= 3) g.appendChild(mk("g-armL"));
  if(wrongCount >= 4) g.appendChild(mk("g-armR"));
  if(wrongCount >= 5) g.appendChild(mk("g-legL"));
  if(wrongCount >= 6) g.appendChild(mk("g-legR"));

  resizeLayout();
}

/* ==== GAME ==== */
function newGame(){
  secret = (WORDS[Math.floor(Math.random()*WORDS.length)] || "hangman").toLowerCase();
  revealed = [];
  for(var i=0;i<secret.length;i++){
    var c = secret.charAt(i);
    revealed.push((c>='a'&&c<='z') ? "_" : c);
  }
  guessed = {};
  wrongCount = 0;
  setText("wrongCount","0");
  setText("maxWrong", String(MAX_WRONG));
  setText("guessedList","—");

  buildKeyboard();
  drawGallows();
  drawWord();
  setMsg("");

  setTimeout(resizeLayout, 40); // second pass for some firmwares
}

function drawWord(){
  var s = "";
  for(var i=0;i<revealed.length;i++){ s += revealed[i] + " "; }
  setText("wordBox", s.replace(/\s+$/,""));
}

function onLetterPress(letter){
  if(!letter) return;
  if(guessed[letter]) return;
  guessed[letter] = true;
  var btn = $("key_" + letter.toUpperCase());
  if(btn){ try{ btn.disabled = true; }catch(e){} }

  // guessed list
  var list = [];
  for(var k in guessed){ if(guessed.hasOwnProperty(k)) list.push(k.toUpperCase()); }
  setText("guessedList", list.length ? list.join(", ") : "—");

  var found = false, i;
  for(i=0;i<secret.length;i++){
    if(secret.charAt(i) === letter){ revealed[i] = letter.toUpperCase(); found = true; }
  }

  if(found){
    drawWord();
    if(isRevealed()){ setMsg("You win! " + secret.toUpperCase()); disableAll(); }
  }else{
    wrongCount++;
    setText("wrongCount", String(wrongCount));
    drawGallows();
    if(wrongCount >= MAX_WRONG){
      revealAll();
      setMsg("You lost — word: " + secret.toUpperCase());
      disableAll();
    }
  }
}

function isRevealed(){ for(var i=0;i<revealed.length;i++){ if(revealed[i]==="_") return false; } return true; }
function revealAll(){ for(var i=0;i<secret.length;i++){ revealed[i] = secret.charAt(i).toUpperCase(); } drawWord(); }
function disableAll(){ var i, ch, b; for(i=65;i<=90;i++){ ch=String.fromCharCode(i); b=$("key_"+ch); if(b){ try{ b.disabled=true; }catch(e){} } } }

/* Buttons */
function press(btn, action){
  try{ if(btn && btn.blur) btn.blur(); }catch(e){}
  if(action==="new") return newGame();
  if(action==="refresh") return refreshScreen();
}

/* Force a full-screen repaint without changing state */
function refreshScreen(){
  var ov = document.createElement("div");
  ov.style.position="fixed"; ov.style.left="0"; ov.style.top="0";
  ov.style.width="100%"; ov.style.height="100%"; ov.style.background="#ffffff"; ov.style.zIndex="9999";
  document.body.appendChild(ov);
  setTimeout(function(){ if(ov && ov.parentNode){ ov.parentNode.removeChild(ov); } }, 160);
}

/* ==== LAYOUT ==== */
function resizeLayout(){
  try{
    var w = window.innerWidth || document.documentElement.clientWidth || 600;
    var h = window.innerHeight || document.documentElement.clientHeight || 800;

    /* Top sizes (keep modest, like your previous games) */
    var base = Math.floor(Math.min(w,h) / 28); // smaller base
    var titleFS = clamp(Math.floor(base * 1.2), 20, 26);   // ~24 on Kindle
    var subFS   = clamp(Math.floor(base * 0.6), 11, 13);
    var chipFS  = clamp(Math.floor(base * 0.6), 11, 13);

    $("title").style.fontSize    = titleFS + "px";
    $("subtitle").style.fontSize = subFS + "px";

    var chips = document.getElementsByClassName("chip");
    var i;
    for(i=0;i<chips.length;i++){
      chips[i].style.fontSize = chipFS + "px";
      chips[i].style.padding  = "3px 8px";
    }

    /* Gallows size: ~62% width, capped by 54% height; square */
    var gW = Math.floor(w * 0.62), gH = Math.floor(h * 0.54);
    var side = gW < gH ? gW : gH;
    if(side < 180) side = 180;
    var g = $("gallows");
    if(g){ g.style.width = side + "px"; g.style.height = side + "px"; }

    /* Proportions from side */
    var head = Math.floor(side * 0.12);
    var border = Math.max(2, Math.floor(head * 0.18));
    var rope = Math.floor(side * 0.075);
    var bodyW = Math.max(4, Math.floor(side * 0.034));
    var bodyH = Math.floor(side * 0.25);
    var armL  = Math.floor(side * 0.20);
    var armT  = Math.max(4, Math.floor(side * 0.022));
    var legW  = armT, legH = Math.floor(side * 0.20);

    var poleX = Math.floor(side * 0.16);
    var beamLen = Math.floor(side * 0.58);
    var ropeX = poleX + beamLen - 4;

    /* Base */
    var baseEl = $("g-base");
    if(baseEl){
      baseEl.style.left   = Math.floor(side * 0.06) + "px";
      baseEl.style.bottom = "6px";
      baseEl.style.width  = Math.floor(side * 0.84) + "px";
      baseEl.style.height = "6px";
      baseEl.style.background = "#111";
    }

    /* Pole */
    var pole = $("g-pole");
    if(pole){
      pole.style.left   = poleX + "px";
      pole.style.bottom = "6px";
      pole.style.width  = "6px";
      pole.style.height = Math.floor(side * 0.70) + "px";
      pole.style.background = "#111";
    }

    /* Beam */
    var beam = $("g-beam");
    if(beam){
      beam.style.left   = poleX + "px";
      beam.style.top    = "6px";
      beam.style.width  = beamLen + "px";
      beam.style.height = "6px";
      beam.style.background = "#111";
    }

    /* Rope */
    var ropeEl = $("g-rope");
    if(ropeEl){
      ropeEl.style.left   = ropeX + "px";
      ropeEl.style.top    = "6px";
      ropeEl.style.width  = "2px";
      ropeEl.style.height = rope + "px";
      ropeEl.style.background = "#333";
    }

    /* Head */
    var headEl = $("g-head");
    if(headEl){
      headEl.style.left         = (ropeX - Math.floor(head/2)) + "px";
      headEl.style.top          = (6 + rope) + "px";
      headEl.style.width        = head + "px";
      headEl.style.height       = head + "px";
      headEl.style.borderRadius = Math.floor(head/2) + "px";
      headEl.style.border       = border + "px solid #111";
      headEl.style.background   = "#fff";
      headEl.style.boxSizing    = "border-box";
    }

    /* Body */
    var bodyTop = 6 + rope + head;
    var bodyEl = $("g-body");
    if(bodyEl){
      bodyEl.style.left   = (ropeX - Math.floor(bodyW/2)) + "px";
      bodyEl.style.top    = bodyTop + "px";
      bodyEl.style.width  = bodyW + "px";
      bodyEl.style.height = bodyH + "px";
      bodyEl.style.background = "#111";
    }

    /* Arms */
    var armY = bodyTop + Math.floor(bodyH * 0.30);
    var armLEl = $("g-armL");
    if(armLEl){
      armLEl.style.left   = (ropeX - armL) + "px";
      armLEl.style.top    = armY + "px";
      armLEl.style.width  = armL + "px";
      armLEl.style.height = armT + "px";
      armLEl.style.background = "#111";
    }
    var armREl = $("g-armR");
    if(armREl){
      armREl.style.left   = (ropeX + 2) + "px";
      armREl.style.top    = armY + "px";
      armREl.style.width  = armL + "px";
      armREl.style.height = armT + "px";
      armREl.style.background = "#111";
    }

    /* Legs */
    var legY = bodyTop + bodyH;
    var legLEl = $("g-legL");
    if(legLEl){
      legLEl.style.left   = (ropeX - Math.floor(legW*2)) + "px";
      legLEl.style.top    = legY + "px";
      legLEl.style.width  = legW + "px";
      legLEl.style.height = legH + "px";
      legLEl.style.background = "#111";
    }
    var legREl = $("g-legR");
    if(legREl){
      legREl.style.left   = (ropeX + legW) + "px";
      legREl.style.top    = legY + "px";
      legREl.style.width  = legW + "px";
      legREl.style.height = legH + "px";
      legREl.style.background = "#111";
    }

    /* Word sizing (single line, modest) */
    var word = $("wordBox");
    if(word){
      var maxW = Math.floor(w * 0.92);
      var charSlots = Math.max(6, secret.length * 2 - 1);
      var fs = clamp(Math.floor(maxW / charSlots), 18, 40);
      word.style.fontSize = fs + "px";
      word.style.letterSpacing = clamp(Math.floor(fs * 0.28), 4, 14) + "px";
    }

    /* Keyboard buttons (similar to your earlier games) */
    var keys = document.getElementsByClassName("letter");
    var keyW = Math.max(34, Math.floor(w / 18));
    var keyH = Math.max(34, Math.floor(h * 0.06));
    for(i=0;i<keys.length;i++){
      keys[i].style.minWidth = keyW + "px";
      keys[i].style.height   = keyH + "px";
      keys[i].style.fontSize = Math.max(14, Math.floor(keyH * 0.48)) + "px";
    }

    /* Action buttons */
    var acts = $("actionsTable").getElementsByTagName("button");
    for(i=0;i<acts.length;i++){
      acts[i].style.minWidth = Math.max(110, Math.floor(w*0.26)) + "px";
      acts[i].style.height   = Math.max(34, Math.floor(h*0.055)) + "px";
      acts[i].style.fontSize = Math.max(15, Math.floor(h*0.02)) + "px";
    }
  }catch(e){}
}

/* Keys for desktop/testing */
function onKey(e){
  try{
    var k = e.keyCode || e.which;
    if(k>=65 && k<=90){
      var ch = String.fromCharCode(k).toLowerCase();
      var b = $("key_" + ch.toUpperCase());
      if(b && !b.disabled) onLetterPress(ch);
    }
    if(k===78) newGame(); // N
  }catch(e){}
}

/* Init */
function init(){
  buildKeyboard();
  newGame();
  try{ window.addEventListener("resize", resizeLayout); }catch(e){}
  try{ document.addEventListener("keydown", onKey); }catch(e){}
  resizeLayout();
}
if(document.readyState==="loading"){
  try{ document.addEventListener("DOMContentLoaded", init); }catch(e){ window.onload = init; }
}else{
  init();
}

/* expose for inline */
try{ window.press = press; }catch(e){}