/* Connect 4 for Illusion — ES5-only, updated to use .disc inner elements (black/gray) */

/* same constants as before */
var COLS = 7;
var ROWS = 6;

/* Board state: 0 empty, 1 black, 2 gray */
var board = [];
var turn = 1; // 1 = Black, 2 = Gray
var gameOverFlag = false;
var undoStack = [];
var scoreB = 0, scoreG = 0;

/* DOM helpers */
function $(id){ try{ return document.getElementById(id); }catch(e){ return null; } }
function setText(id, txt){ var el=$(id); if(!el) return; try{ el.textContent = txt; }catch(e){ el.innerHTML = txt; } }
function addClass(el, name){ if(!el) return; if((' '+el.className+' ').indexOf(' '+name+' ' )===-1) el.className = (el.className?el.className+' ':'') + name; }
function removeClass(el, name){ if(!el) return; el.className = (' '+el.className+' ').replace(' '+name+' ', ' ').replace(/^\s+|\s+$/g,''); }

/* Build board HTML once: td contains a div.disc */
function buildBoard(){
  var t = $("board");
  t.innerHTML = "";
  var r, c, tr, td, disc;
  for(r=0; r<ROWS; r++){
    tr = document.createElement("tr");
    for(c=0; c<COLS; c++){
      td = document.createElement("td");
      td.dataset.r = r;
      td.dataset.c = c;
      // create disc element
      disc = document.createElement("div");
      disc.className = "disc disc-empty";
      td.appendChild(disc);
      tr.appendChild(td);
    }
    t.appendChild(tr);
  }

  // column buttons
  var cb = $("colButtons");
  cb.innerHTML = "";
  var tr2 = document.createElement("tr");
  for(c=0; c<COLS; c++){
    var td2 = document.createElement("td");
    var btn = document.createElement("button");
    btn.type = "button";
    btn.className = "colBtn";
    btn.id = "colBtn_" + c;
    btn.innerHTML = (c+1).toString();
    (function(col){
      btn.onclick = function(){ dropInColumn(col); };
    })(c);
    td2.appendChild(btn);
    tr2.appendChild(td2);
  }
  cb.appendChild(tr2);
}

/* Initialize board array and start new game */
function newGame(){
  board = [];
  for(var r=0; r<ROWS; r++){
    board[r] = [];
    for(var c=0; c<COLS; c++){ board[r][c] = 0; }
  }
  turn = 1;
  gameOverFlag = false;
  undoStack = [];
  setText("message", "");
  setText("turnName", turn===1 ? "Black" : "Gray");
  updateScoresDisplay();
  drawBoard();
  enableColumnButtons();
  resizeLayout();
}

/* Draw board by updating each .disc element class (fast and resilient) */
function drawBoard(){
  var rows = $("board").getElementsByTagName("tr");
  for(var r=0; r<ROWS; r++){
    var cells = rows[r].getElementsByTagName("td");
    for(var c=0; c<COLS; c++){
      var td = cells[c];
      var disc = td.firstChild;
      if(!disc) continue;
      // reset win outline on td
      removeClass(td, "cell-win");
      if(board[r][c] === 1){
        disc.className = "disc disc-black";
      } else if(board[r][c] === 2){
        disc.className = "disc disc-gray";
      } else {
        disc.className = "disc disc-empty";
      }
    }
  }
}

/* Drop piece in column (if valid) */
function dropInColumn(col){
  if(gameOverFlag) return;
  // find bottom-most empty row
  var r;
  for(r=ROWS-1; r>=0; r--){
    if(board[r][col] === 0) break;
  }
  if(r < 0) {
    setText("message", "Column full");
    return;
  }

  // push undo snapshot (simple: store row, col)
  undoStack.push({r:r, c:col, prev:0});
  board[r][col] = turn;
  setText("message", "");
  drawBoard();

  // check win
  var winCells = checkWinFrom(r, col, turn);
  if(winCells && winCells.length >= 4){
    gameOverFlag = true;
    // highlight win cells
    highlightWin(winCells);
    setText("message", (turn===1?"Black":"Gray") + " wins!");
    if(turn===1) scoreB++; else scoreG++;
    updateScoresDisplay();
    disableColumnButtons();
    return;
  }

  // check draw
  if(isBoardFull()){
    gameOverFlag = true;
    setText("message", "Draw");
    disableColumnButtons();
    return;
  }

  // switch turn
  turn = (turn===1) ? 2 : 1;
  setText("turnName", turn===1 ? "Black" : "Gray");
}

/* Undo last move */
function undo(){
  if(!undoStack.length || gameOverFlag) return;
  var a = undoStack.pop();
  board[a.r][a.c] = a.prev;
  drawBoard();
  setText("message", "Undo");
  // switch back turn
  turn = (turn===1) ? 2 : 1;
  setText("turnName", turn===1 ? "Black" : "Gray");
  enableColumnButtons();
}

/* Check board full */
function isBoardFull(){
  for(var c=0; c<COLS; c++){
    if(board[0][c] === 0) return false;
  }
  return true;
}

/* Check for 4-in-a-row starting from a placed disc (checks all directions).
   Returns array of winning cell coords [{r:,c:},...] or null */
function checkWinFrom(r0, c0, player){
  var dirs = [
    {dr:0, dc:1},  // horizontal
    {dr:1, dc:0},  // vertical
    {dr:1, dc:1},  // diag down-right
    {dr:1, dc:-1}  // diag down-left
  ];

  for(var d=0; d<dirs.length; d++){
    var dr = dirs[d].dr, dc = dirs[d].dc;
    var hits = [{r:r0, c:c0}];

    // scan negative direction
    var r = r0 - dr, c = c0 - dc;
    while(r>=0 && r<ROWS && c>=0 && c<COLS && board[r][c] === player){
      hits.unshift({r:r,c:c});
      r -= dr; c -= dc;
    }
    // scan positive direction
    r = r0 + dr; c = c0 + dc;
    while(r>=0 && r<ROWS && c>=0 && c<COLS && board[r][c] === player){
      hits.push({r:r,c:c});
      r += dr; c += dc;
    }

    if(hits.length >= 4) return hits;
  }
  return null;
}

/* Highlight winning cells: add .cell-win to TD and keep disc class as-is */
function highlightWin(cells){
  var rows = $("board").getElementsByTagName("tr");
  for(var i=0; i<cells.length; i++){
    var rr = cells[i].r, cc = cells[i].c;
    var td = rows[rr].getElementsByTagName("td")[cc];
    if(td) addClass(td, "cell-win");
  }
}

/* Enable/disable column buttons */
function disableColumnButtons(){
  for(var c=0; c<COLS; c++){
    var b = $("colBtn_" + c);
    if(b) b.disabled = true;
  }
}
function enableColumnButtons(){
  for(var c=0; c<COLS; c++){
    var b = $("colBtn_" + c);
    if(b) b.disabled = false;
  }
}

/* Refresh screen without reloading (overlay trick) */
function refreshScreen(){
  var ov = document.createElement("div");
  ov.id = "refreshOverlay";
  ov.style.position = "fixed"; ov.style.left = "0"; ov.style.top = "0";
  ov.style.width = "100%"; ov.style.height = "100%";
  ov.style.background = "#ffffff"; ov.style.zIndex = "9999";
  document.body.appendChild(ov);
  setTimeout(function(){
    if(ov && ov.parentNode) ov.parentNode.removeChild(ov);
  }, 160);
}

/* Buttons wrapper */
function press(btn, action){
  try{ if(btn && btn.blur) btn.blur(); }catch(e){}
  if(action === "new"){ newGame(); enableColumnButtons(); return; }
  if(action === "undo"){ undo(); return; }
  if(action === "refresh"){ refreshScreen(); return; }
}

/* Keyboard support for desktop testing */
function onKey(e){
  try{
    var k = e.keyCode || e.which;
    // 49..55 = keys 1..7 to drop in columns
    if(k >= 49 && k <= 49 + (COLS-1)){
      var col = k - 49;
      dropInColumn(col);
    }
    if(k === 78){ newGame(); } // N
    if(k === 85){ undo(); } // U
  }catch(e){}
}

/* Resize layout to fit screen without scrolling */
function resizeLayout(){
  try{
    var w = window.innerWidth || document.documentElement.clientWidth || 600;
    var h = window.innerHeight || document.documentElement.clientHeight || 800;

    // Elements that take vertical space: top, controls (colButtons), actions, message
    var topEl = $("top");
    var ctrl = $("colButtons");
    var acts = $("actionsTable");
    var msg = $("message");

    var topH = topEl ? topEl.offsetHeight : 0;
    var ctrlH = ctrl ? ctrl.offsetHeight : 0;
    var actH = acts ? acts.offsetHeight : 0;
    var msgH = msg ? msg.offsetHeight : 0;

    var extra = 16; // margins
    var availableH = h - (topH + ctrlH + actH + msgH + extra);
    if(availableH < 120) availableH = 120;

    var gridMaxW = Math.floor(w * 0.94);
    var gridMaxH = Math.floor(availableH);

    // cell size
    var cellW = Math.floor(gridMaxW / COLS);
    var cellH = Math.floor(gridMaxH / ROWS);
    var cell = Math.max(28, Math.min(cellW, cellH));

    // apply to td cells and inner disc size (disc is ~80% of cell)
    var tds = $("board").getElementsByTagName("td");
    for(var i=0; i<tds.length; i++){
      tds[i].style.width = cell + "px";
      tds[i].style.height = cell + "px";
      // set disc exact px sizes for consistent look
      var disc = tds[i].firstChild;
      if(disc){
        var d = Math.max(12, Math.floor(cell * 0.78));
        disc.style.width = d + "px";
        disc.style.height = d + "px";
        disc.style.borderRadius = Math.floor(d/2) + "px";
      }
    }

    // column buttons sizing
    for(var c=0; c<COLS; c++){
      var b = $("colBtn_" + c);
      if(b){
        b.style.width = Math.max(28, Math.floor(cell * 0.9)) + "px";
        b.style.height = Math.max(30, Math.floor(cell * 0.9)) + "px";
        b.style.fontSize = Math.max(14, Math.floor(cell * 0.45)) + "px";
      }
    }

    // title and chips scale a bit
    var base = Math.max(14, Math.floor(cell / 2.5));
    $("title").style.fontSize = Math.max(18, Math.min(24, base + 6)) + "px";
    $("subtitle").style.fontSize = Math.max(11, Math.floor(base * 0.6)) + "px";
    var chips = document.getElementsByClassName("chip");
    for(i=0;i<chips.length;i++){
      chips[i].style.fontSize = Math.max(11, Math.floor(base * 0.6)) + "px";
      chips[i].style.padding = "3px 8px";
    }
  }catch(e){}
}

/* Update score display */
function updateScoresDisplay(){
  setText("scoreR", String(scoreB));
  setText("scoreY", String(scoreG));
}

/* Init */
function init(){
  buildBoard();
  newGame();
  try{ window.addEventListener("resize", resizeLayout); }catch(e){}
  try{ document.addEventListener("keydown", onKey); }catch(e){}
  resizeLayout();
}

/* Expose globals used in inline onclicks */
try{ window.dropInColumn = dropInColumn; window.press = press; }catch(e){}

/* Start */
if(document.readyState === "loading"){
  try{ document.addEventListener("DOMContentLoaded", init); }catch(e){ window.onload = init; }
}else{ init(); }